
// __________________________________________________________
//
//                         sbs_pdb.c
//             SBS Program Database Parser V1.00
//                06-25-2006 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#define  _SBS_PDB_DLL_
#include "sbs_pdb.h"

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantibility and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

06-25-2006 V1.00 Original version (SBS).

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

HINSTANCE ghInstance = NULL;

// =================================================================
// GLOBAL STRINGS
// =================================================================

BYTE gabCaption [] = SA(MAIN_CAPTION);
WORD gawCaption [] = SW(MAIN_CAPTION);

// =================================================================
// SUPPORTED VERSIONS
// =================================================================

PDB_VERSION gaVersions [] =
    {
    PDB_VERSION_200, PDB_SIGNATURE_200_, PDB_SIGNATURE_200,
    PDB_VERSION_700, PDB_SIGNATURE_700_, PDB_SIGNATURE_700,
    0
    };

// =================================================================
// PDB FILE MANAGEMENT
// =================================================================

DWORD WINAPI pdbFileValid (PPDB_HEADER pph,
                           DWORD       dFileBytes)
    {
    PPDB_VERSION ppv;
    DWORD        dFilePages, dPageBytes, i;
    DWORD        dVersion = 0;

    if (pph != NULL)
        {
        for (i = 0; (ppv = gaVersions + i)->dVersion; i++)
            {
            if ((dFileBytes >= ppv->dHeader) &&
                (!_tcompA (pph->abSignature, ppv->pbSignature)))
                {
                dPageBytes = 0;

                switch (ppv->dVersion)
                    {
                    case PDB_VERSION_200:
                        {
                        dFilePages = pph->V200.wFilePages;
                        dPageBytes = pph->V200.dPageBytes;
                        break;
                        }
                    case PDB_VERSION_700:
                        {
                        dFilePages = pph->V700.dFilePages;
                        dPageBytes = pph->V700.dPageBytes;
                        break;
                        }
                    }
                if (dPageBytes &&
                    (dFilePages * dPageBytes == dFileBytes))
                    {
                    dVersion = ppv->dVersion;
                    break;
                    }
                }
            }
        }
    return dVersion;
    }

// -----------------------------------------------------------------

PPDB_HEADER WINAPI pdbFileHeaderA (PBYTE  pbPath,
                                   PDWORD pdVersion,
                                   PBOOL  pfInvalid)
    {
    DWORD       dFileBytes;
    DWORD       dVersion = 0;
    BOOL        fInvalid = FALSE;
    PPDB_HEADER pph      = _floadA (pbPath, &dFileBytes);

    if ((pph != NULL) &&
        (fInvalid = !(dVersion = pdbFileValid (pph, dFileBytes))))
        {
        pph = _mfree (pph);
        }
    if (pdVersion != NULL) *pdVersion = dVersion;
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return pph;
    }

// -----------------------------------------------------------------

PPDB_HEADER WINAPI pdbFileHeaderW (PWORD  pwPath,
                                   PDWORD pdVersion,
                                   PBOOL  pfInvalid)
    {
    DWORD       dFileBytes;
    DWORD       dVersion = 0;
    BOOL        fInvalid = FALSE;
    PPDB_HEADER pph      = _floadW (pwPath, &dFileBytes);

    if ((pph != NULL) &&
        (fInvalid = !(dVersion = pdbFileValid (pph, dFileBytes))))
        {
        pph = _mfree (pph);
        }
    if (pdVersion != NULL) *pdVersion = dVersion;
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return pph;
    }

// -----------------------------------------------------------------

DWORD WINAPI pdbFileLimit (PPDB_HEADER pph,
                           DWORD       dVersion)
    {
    DWORD dLimit = 0;

    if (pph != NULL)
        {
        switch (dVersion)
            {
            case PDB_VERSION_200:
                {
                dLimit = ((DWORD) pph->V200.wStartPage - 1) *
                         pph->V200.dPageBytes * 8 *
                         pph->V200.dPageBytes;
                break;
                }
            case PDB_VERSION_700:
                {
                dLimit = pph->V700.dPageBytes * 8 *
                         pph->V700.dPageBytes;
                break;
                }
            }
        }
    return dLimit;
    }

// -----------------------------------------------------------------

BOOL WINAPI pdbFileUnused (PPDB_HEADER pph,
                           DWORD       dVersion,
                           DWORD       dStreamBytes,
                           PDWORD      pdPageBytes)
    {
    DWORD dPageBytes = 0;
    BOOL  fUnused    = FALSE;

    if (pph != NULL)
        {
        switch (dVersion)
            {
            case PDB_VERSION_200:
                {
                if (!(fUnused = (dStreamBytes == PDB_UNUSED_16)))
                    {
                    dPageBytes = pph->V200.dPageBytes;
                    }
                break;
                }
            case PDB_VERSION_700:
                {
                if (!(fUnused = (dStreamBytes == PDB_UNUSED_32)))
                    {
                    dPageBytes = pph->V700.dPageBytes;
                    }
                break;
                }
            }
        }
    if (pdPageBytes != NULL) *pdPageBytes = dPageBytes;
    return fUnused;
    }

// -----------------------------------------------------------------

DWORD WINAPI pdbFilePages (PPDB_HEADER pph,
                           DWORD       dVersion,
                           DWORD       dStreamBytes,
                           PDWORD      pdPageBytes,
                           PBOOL       pfUnused)
    {
    DWORD dPageBytes = 0;
    BOOL  fUnused    = pdbFileUnused (pph, dVersion, dStreamBytes,
                                      &dPageBytes);

    if (pdPageBytes != NULL) *pdPageBytes = dPageBytes;
    if (pfUnused    != NULL) *pfUnused    = fUnused;

    return (dPageBytes && dStreamBytes
            ? ((dStreamBytes - 1) / dPageBytes) + 1
            : 0);
    }

// -----------------------------------------------------------------

PVOID WINAPI pdbFileRead (PPDB_HEADER pph,
                          DWORD       dVersion,
                          DWORD       dStreamBytes,
                          PVOID       pPages,
                          PBOOL       pfUnused)
    {
    DWORD dPageBytes, dPages, i, j, n;
    BOOL  fUnused = FALSE;
    PVOID pStream = NULL;

    if (pPages != NULL)
        {
        dPages = pdbFilePages (pph, dVersion, dStreamBytes,
                               &dPageBytes, &fUnused);
        if (dPageBytes &&
            ((pStream = _mnew (n = dStreamBytes)) != NULL))
            {
            for (i = 0; i < dPages; i++)
                {
                switch (dVersion)
                    {
                    case PDB_VERSION_200:
                        {
                        j = ((PWORD ) pPages) [i];
                        break;
                        }
                    case PDB_VERSION_700:
                        {
                        j = ((PDWORD) pPages) [i];
                        break;
                        }
                    }
                _mcopy (SKIP (pStream, i * dPageBytes),
                        SKIP (pph,     j * dPageBytes),
                        min (n, dPageBytes));

                n -= dPageBytes;
                }
            }
        }
    if (pfUnused != NULL) *pfUnused = fUnused;
    return pStream;
    }

// -----------------------------------------------------------------

PPDB_ROOT WINAPI pdbFileRoot (PPDB_HEADER pph,
                              DWORD       dVersion,
                              PDWORD      pdStreams,
                              PDWORD      pdStreamBytes,
                              PBOOL       pfUnused)
    {
    DWORD     dIndexBytes;
    PVOID     pPages       = NULL;
    DWORD     dStreams     = 0;
    DWORD     dStreamBytes = 0;
    BOOL      fUnused      = FALSE;
    PPDB_ROOT ppr          = NULL;

    if (pph != NULL)
        {
        switch (dVersion)
            {
            case PDB_VERSION_200:
                {
                dStreamBytes = pph->V200.RootStream.dStreamBytes;
                pPages       = pph->V200.awRootPages;

                if ((ppr = pdbFileRead (pph, dVersion,
                                        dStreamBytes, pPages,
                                        &fUnused))
                    != NULL)
                    {
                    dStreams = ppr->V200.wStreams;
                    }
                break;
                }
            case PDB_VERSION_700:
                {
                dStreamBytes = pph->V700.dRootBytes;
                pPages       = pph->V700.adIndexPages;

                dIndexBytes  = pdbFilePages (pph, dVersion,
                                             dStreamBytes,
                                             NULL, NULL)
                               * DWORD_;

                if ((pPages = pdbFileRead (pph, dVersion,
                                           dIndexBytes,  pPages,
                                           &fUnused))
                    != NULL)
                    {
                    if ((ppr = pdbFileRead (pph, dVersion,
                                            dStreamBytes, pPages,
                                            &fUnused))
                        != NULL)
                        {
                        dStreams = ppr->V700.dStreams;
                        }
                    _mfree (pPages);
                    }
                break;
                }
            }
        if (ppr == NULL) dStreamBytes = 0;
        }
    if (pdStreams     != NULL) *pdStreams     = dStreams;
    if (pdStreamBytes != NULL) *pdStreamBytes = dStreamBytes;
    if (pfUnused      != NULL) *pfUnused      = fUnused;
    return ppr;
    }

// -----------------------------------------------------------------

PVOID WINAPI pdbFileStream (PPDB_HEADER pph,
                            PPDB_ROOT   ppr,
                            DWORD       dVersion,
                            DWORD       dStreamID,
                            PDWORD      pdStreamBytes,
                            PBOOL       pfUnused)
    {
    DWORD dPages, dOffset, i;
    DWORD dStreamBytes = 0;
    BOOL  fUnused      = FALSE;
    PVOID pStream      = NULL;

    if ((pph != NULL) && (ppr != NULL))
        {
        switch (dVersion)
            {
            case PDB_VERSION_200:
                {
                if (dStreamID < ppr->V200.wStreams)
                    {
                    for (dPages = i = 0; i < dStreamID; i++)
                        {
                        dStreamBytes = ppr->V200.aStreams [i]
                                                .dStreamBytes;

                        dPages += pdbFilePages (pph, dVersion,
                                                dStreamBytes,
                                                NULL, NULL);
                        }
                    dStreamBytes = ppr->V200.aStreams [i]
                                            .dStreamBytes;

                    dOffset = PDB_ROOT_200__ (ppr->V200.wStreams) +
                              (dPages * WORD_);

                    pStream = pdbFileRead (pph, dVersion,
                                           dStreamBytes,
                                           SKIP (ppr, dOffset),
                                           &fUnused);
                    }
                break;
                }
            case PDB_VERSION_700:
                {
                if (dStreamID < ppr->V700.dStreams)
                    {
                    for (dPages = i = 0; i < dStreamID; i++)
                        {
                        dStreamBytes = ppr->V700.adStreamBytes [i];

                        dPages += pdbFilePages (pph, dVersion,
                                                dStreamBytes,
                                                NULL, NULL);
                        }
                    dStreamBytes = ppr->V700.adStreamBytes [i];

                    dOffset = PDB_ROOT_700__ (ppr->V700.dStreams) +
                              (dPages * DWORD_);

                    pStream = pdbFileRead (pph, dVersion,
                                           dStreamBytes,
                                           SKIP (ppr, dOffset),
                                           &fUnused);
                    }
                break;
                }
            }
        if (pStream == NULL) dStreamBytes = 0;
        }
    if (pdStreamBytes != NULL) *pdStreamBytes = dStreamBytes;
    if (pfUnused      != NULL) *pfUnused      = fUnused;
    return pStream;
    }

// -----------------------------------------------------------------

PPDB_FILE WINAPI pdbFileOpenA (PBYTE pbPath,
                               PBOOL pfInvalid)
    {
    DWORD       n;
    BYTE        abPath [MAX_PATH];
    PBYTE       pbFile;
    BOOL        fInvalid = FALSE;
    PPDB_FILE   ppf      = NULL;

    if ((pbPath != NULL) && pbPath [0] &&
        (n = GetFullPathNameA (pbPath, MAX_PATH, abPath, &pbFile))&&
        (n < MAX_PATH) &&
        ((ppf = _mnew (PDB_FILE_)) != NULL))
        {
        ppf->pHeader = pdbFileHeaderA (abPath,
                                       &ppf->dVersion, &fInvalid);

        if ((ppf->pRoot = pdbFileRoot (ppf->pHeader, ppf->dVersion,
                                       &ppf->dStreams, &ppf->dRoot,
                                       NULL))
            != NULL)
            {
            _mzero    (ppf->awPath, sizeof (ppf->awPath));
            _tcopyA2W (ppf->awPath, abPath, -1);
            }
        else
            {
            ppf->pHeader = _mfree (ppf->pHeader);
            ppf          = _mfree (ppf         );
            }
        }
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return ppf;
    }

// -----------------------------------------------------------------

PPDB_FILE WINAPI pdbFileOpenW (PWORD pwPath,
                               PBOOL pfInvalid)
    {
    DWORD       n;
    WORD        awPath [MAX_PATH];
    PWORD       pwFile;
    BOOL        fInvalid = FALSE;
    PPDB_FILE   ppf      = NULL;

    if ((pwPath != NULL) && pwPath [0] &&
        (n = GetFullPathNameW (pwPath, MAX_PATH, awPath, &pwFile))&&
        (n < MAX_PATH) &&
        ((ppf = _mnew (PDB_FILE_)) != NULL))
        {
        ppf->pHeader = pdbFileHeaderW (awPath,
                                       &ppf->dVersion, &fInvalid);

        if ((ppf->pRoot = pdbFileRoot (ppf->pHeader, ppf->dVersion,
                                       &ppf->dStreams, &ppf->dRoot,
                                       NULL))
            != NULL)
            {
            _mzero    (ppf->awPath, sizeof (ppf->awPath));
            _tcopyW2W (ppf->awPath, awPath, -1);
            }
        else
            {
            ppf->pHeader = _mfree (ppf->pHeader);
            ppf          = _mfree (ppf         );
            }
        }
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return ppf;
    }

// -----------------------------------------------------------------

PPDB_FILE WINAPI pdbFileClose (PPDB_FILE ppf)
    {
    if (ppf != NULL)
        {
        _mfree (ppf->pHeader);
        _mfree (ppf->pRoot  );
        _mfree (ppf         );
        }
    return NULL;
    }

// -----------------------------------------------------------------

PVOID WINAPI pdbFileExtract (PPDB_FILE ppf,
                             DWORD     dStreamID,
                             PDWORD    pdStreamBytes,
                             PBOOL     pfUnused)
    {
    DWORD dStreamBytes = 0;
    BOOL  fUnused      = FALSE;
    PVOID pStream      = NULL;

    if (ppf != NULL)
        {
        pStream = pdbFileStream (ppf->pHeader,  ppf->pRoot,
                                 ppf->dVersion, dStreamID,
                                 &dStreamBytes, &fUnused);
        }
    if (pdStreamBytes != NULL) *pdStreamBytes = dStreamBytes;
    if (pfUnused      != NULL) *pfUnused      = fUnused;
    return pStream;
    }

// -----------------------------------------------------------------

PPDB_DATA WINAPI pdbFileData (PPDB_FILE ppf)
    {
    DWORD       dPages, dOffset, dStreamBytes, i;
    PPDB_ROOT   ppr;
    PPDB_STREAM pps;
    PPDB_DATA   ppd = NULL;

    if ((ppf != NULL) &&
        ((ppd = _mnew (PDB_DATA__ (ppf->dStreams))) != NULL))
        {
        ppr = ppf->pRoot;

        _mzero  (ppd->awPath, sizeof (ppd->awPath));
        _tcopyW (ppd->awPath, ppf->awPath);

        ppd->dVersion = ppf->dVersion;
        ppd->dStreams = ppf->dStreams;

        switch (ppf->dVersion)
            {
            case PDB_VERSION_200:
                {
                for (dPages = i = 0; i < ppf->dStreams; i++)
                    {
                    pps = ppd->aStreams + i;

                    dStreamBytes = ppr->V200.aStreams [i]
                                            .dStreamBytes;

                    dOffset = PDB_ROOT_200__ (ppr->V200.wStreams) +
                              (dPages * WORD_);

                    pps->pData = pdbFileRead (ppf->pHeader,
                                              ppf->dVersion,
                                              dStreamBytes,
                                              SKIP (ppr, dOffset),
                                              &pps->fUnused);

                    pps->dData = (pps->fUnused ? 0 : dStreamBytes);

                    dPages += pdbFilePages (ppf->pHeader,
                                            ppf->dVersion,
                                            dStreamBytes,
                                            NULL, NULL);
                    }
                break;
                }
            case PDB_VERSION_700:
                {
                for (dPages = i = 0; i < ppf->dStreams; i++)
                    {
                    pps = ppd->aStreams + i;

                    dStreamBytes = ppr->V700.adStreamBytes [i];

                    dOffset = PDB_ROOT_700__ (ppr->V700.dStreams) +
                              (dPages * DWORD_);

                    pps->pData = pdbFileRead (ppf->pHeader,
                                              ppf->dVersion,
                                              dStreamBytes,
                                              SKIP (ppr, dOffset),
                                              &pps->fUnused);

                    pps->dData = (pps->fUnused ? 0 : dStreamBytes);

                    dPages += pdbFilePages (ppf->pHeader,
                                            ppf->dVersion,
                                            dStreamBytes,
                                            NULL, NULL);
                    }
                break;
                }
            }
        }
    return ppd;
    }

// =================================================================
// PDB DATA MANAGEMENT
// =================================================================

PPDB_DATA WINAPI pdbDataOpenA (PBYTE pbPath,
                               PBOOL pfInvalid)
    {
    PPDB_FILE ppf;
    BOOL      fInvalid = FALSE;
    PPDB_DATA ppd      = NULL;

    if ((ppf = pdbFileOpenA (pbPath, &fInvalid)) != NULL)
        {
        ppd = pdbFileData (ppf);
        pdbFileClose (ppf);
        }
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return ppd;
    }

// -----------------------------------------------------------------

PPDB_DATA WINAPI pdbDataOpenW (PWORD pwPath,
                               PBOOL pfInvalid)
    {
    PPDB_FILE ppf;
    BOOL      fInvalid = FALSE;
    PPDB_DATA ppd      = NULL;

    if ((ppf = pdbFileOpenW (pwPath, &fInvalid)) != NULL)
        {
        ppd = pdbFileData (ppf);
        pdbFileClose (ppf);
        }
    if (pfInvalid != NULL) *pfInvalid = fInvalid;
    return ppd;
    }

// -----------------------------------------------------------------

PPDB_DATA WINAPI pdbDataClose (PPDB_DATA ppd)
    {
    DWORD i;

    if (ppd != NULL)
        {
        for (i = 0; i < ppd->dStreams; i++)
            {
            _mfree (ppd->aStreams [i].pData);
            }
        _mfree (ppd);
        }
    return NULL;
    }

// =================================================================
// DLL MANAGEMENT
// =================================================================

BOOL WINAPI DllMain (HINSTANCE hInstance,
                     DWORD     dReason,
                     PVOID     pReserved)
    {
    BOOL fOk = TRUE;

    if (dReason == DLL_PROCESS_ATTACH)
        {
        ghInstance = hInstance;
        }
    return fOk;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
